#!/usr/bin/perl -w

# Program gptparser.pl

# (c) Gary C. Kessler, 2011 [gck@garykessler.net]

# Program to parse Globally Unique Identifier (GUID) Partition
# Table (GPT) data structures.
# Ref: File System Forensic Analysis, Brian Carrier
#  and http://en.wikipedia.org/wiki/GUID_Partition_Table

# General structure of the GPT
#
# SECTOR 0: Legacy (or Protective) MBR
# SECTOR 1: GPT header
# SECTORS 2-33: Partition table entries

# PROGRAMMER'S MEA CULPA.....
# I use GOTOs in this program. I actually make no apologies. I know that I
# could have used if.then.else statements. And the indentation would have made
# the code harder to figure out than my using the GOTOs in lieu of a BREAK. I
# think that it actually makes the program more readable and the logic easier
# to follow.


# ********************************************************
# MAIN PROGRAM BLOCK
# ********************************************************

# Initialize global variables

$version = "1.2 beta";
$build_date = "6 June 2012";

# $source is the name of the file to read.

$source = "";

# partition_type is an array with a small set of partition type GUID
# values. Ref: http://en.wikipedia.org/wiki/GUID_Partition_Table

%partition_type =
  (
  "00000000-0000-0000-0000-000000000000" => "Unused entry",
  "024DEE41-33E7-11D3-9D69-0008C781F39F" => "MBR partition scheme",
  "0657FD6D-A4AB-43C4-84E5-0933C84B4F4F" => "Swap partition (Linux)",
  "21686148-6449-6E6F-744E-656564454649" => "BIOS Boot partition",
  "2DB519C4-B10F-11DC-B99B-0019D1879648" => "Concatenated partition (NetBSD)",
  "2DB519EC-B10F-11DC-B99B-0019D1879648" => "Encrypted partition (NetBSD)",
  "2E0A753D-9E48-43B0-8337-B15192CB1B5E" => "ChromeOS future use",
  "37AFFC90-EF7D-4E96-91C3-2D7AE055B174" => "IBM General Parallel File System (GPFS) partition (Windows)",
  "3CB8E202-3B7E-47DD-8A3C-7FF2A13CFCEC" => "ChromeOS rootfs",
  "48465300-0000-11AA-AA11-00306543ECAC" => "Hierarchical File System Plus (HFS+) partition (Mac OS X)",
  "426F6F74-0000-11AA-AA11-00306543ECAC" => "Apple Boot partition (Mac OS X)",
  "49F48D32-B10E-11DC-B99B-0019D1879648" => "Swap partition (NetBSD)",
  "49F48D5A-B10E-11DC-B99B-0019D1879648" => "FFS partition (NetBSD)",
  "49F48D82-B10E-11DC-B99B-0019D1879648" => "LFS partition (NetBSD)",
  "49F48DAA-B10E-11DC-B99B-0019D1879648" => "RAID partition (NetBSD)",
  "4C616265-6C00-11AA-AA11-00306543ECAC" => "Apple Label (Mac OS X)",
  "516E7CB4-6ECF-11D6-8FF8-00022D09712B" => "Data partition (FreeBSD)",
  "516E7CB5-6ECF-11D6-8FF8-00022D09712B" => "Swap partition (FreeBSD)",
  "516E7CB6-6ECF-11D6-8FF8-00022D09712B" => "Unix File System (UFS) partition (FreeBSD)",
  "516E7CB8-6ECF-11D6-8FF8-00022D09712B" => "Vinum volume manager partition (FreeBSD)",
  "516E7CBA-6ECF-11D6-8FF8-00022D09712B" => "ZFS partition (FreeBSD)",
  "52414944-0000-11AA-AA11-00306543ECAC" => "Apple RAID partition (Mac OS X)",
  "52414944-5F4F-11AA-AA11-00306543ECAC" => "Apple RAID partition, offline (Mac OS X)",
  "5265636F-7665-11AA-AA11-00306543ECAC" => "Apple TV Recovery partition (Mac OS X)",
  "55465300-0000-11AA-AA11-00306543ECAC" => "Apple UFS (Mac OS X)",
  "5808C8AA-7E8F-42E0-85D2-E1E90434CFB3" => "Logical Disk Manager metadata partition (Windows)",
  "6A82CB45-1DD2-11B2-99A6-080020736631" => "Boot partition (Solaris)",
  "6A85CF4D-1DD2-11B2-99A6-080020736631" => "Root partition (Solaris)",
  "6A87C46F-1DD2-11B2-99A6-080020736631" => "Swap partition (Solaris)",
  "6A898CC3-1DD2-11B2-99A6-080020736631" => "ZFS (Mac OS X) *or* /usr partition (Solaris)",
  "6A8B642B-1DD2-11B2-99A6-080020736631" => "Backup partition (Solaris)",
  "6A8D2AC7-1DD2-11B2-99A6-080020736631" => "Reserved partition (Solaris)",
  "6A8EF2E9-1DD2-11B2-99A6-080020736631" => "/var partition (Solaris)",
  "6A90BA39-1DD2-11B2-99A6-080020736631" => "/home partition (Solaris)",
  "6A9283A5-1DD2-11B2-99A6-080020736631" => "Alternate sector (Solaris)",
  "6A945A3B-1DD2-11B2-99A6-080020736631" => "Reserved partition (Solaris)",
  "6A9630D1-1DD2-11B2-99A6-080020736631" => "Reserved partition (Solaris)",
  "6A980767-1DD2-11B2-99A6-080020736631" => "Reserved partition (Solaris)",
  "6A96237F-1DD2-11B2-99A6-080020736631" => "Reserved partition (Solaris)",
  "75894C1E-3AEB-11D3-B7C1-7B03A0000000" => "Data partition (HP-UX)",
  "83BD6B9D-7F41-11DC-BE0B-001560B84F0F" => "Boot partition (FreeBSD)",
  "8DA63339-0007-60C0-C436-083AC8230908" => "Reserved (Linux)",
  "A19D880F-05FC-4D3B-A006-743F0F84911E" => "RAID partition (Linux)",
  "AF9B60A0-1431-4F62-BC68-3311714A69AD" => "Logical Disk Manager data partition (Windows)",
  "C12A7328-F81F-11D2-BA4B-00A0C93EC93B" => "EFI System partition",
  "DE94BBA4-06D1-4D40-A16A-BFD50179D6AC" => "Windows Recovery Environment",
  "E2A1E728-32E3-11D6-A682-7B03A0000000" => "Service Partition (HP-UX)",
  "E3C9E316-0B5C-4DB8-817D-F92DF00215AE" => "Microsoft Reserved Partition (Windows)",
  "E6D6D379-F507-44C2-A23C-238F2A3DF928" => "Logical Volume Manager (LVM) partition (Linux)",
  "EBD0A0A2-B9E5-4433-87C0-68B6B72699C7" => "Data partition (Linux *or* Windows)",
  "FE3A2A5D-4F32-41A7-B725-ACCC3285A309" => "ChromeOS kernel",
  );

#
# Begin program activity!!
#

print "\nGPT Parser V$version - Gary C. Kessler ($build_date)\n\n";

# Parse command for file name and request missing information.

if (parsecommandline ())
  { exit 1; };

print "Source file = $source\n";

# Open input ($source) file and stop program if error

open (INFILE, "<:raw", $source)
  or die "Can't open input file \"$source\": $!";
binmode INFILE;

# Start by reading the entire file, which should be no more than 17,408 bytes
# (34 sectors) in length

$header_len = 0;
while (not (eof INFILE) && $header_len < 17408)
  {
  read (INFILE,$byte,1);
  $header [$header_len] = unpack ("C", $byte);
  $header_len++;
  }

close INFILE;
print "Input file length = $header_len bytes.\n\n";

if ($header_len < 512)
  { die "File is smaller than one sector ($len bytes)\n"; }

# File may start with LBA 0 (Protective MBR) or with LBA 1 (GPT Header).
# Determine which and process accordingly. (Assume LBA 1 if the header
# starts with "EF"

# $lba is the sector number that is currently being processed
# $bbase is the absolute byte position in the header of byte 0 in the sector (LBA)
#   currently being processed
# $bpos is the byte position in the sector; reset to 0 with each new LBA

# $bpos, then is the relative byte position within the LBA, $bbase+$bpos is
#    the absolute byte position within $header

$bbase = 0;

if ($header [0] == 0x45 && $header [1] == 0x46)
    { $first_lba = 1; }
  else
    { $first_lba = 0; }


# If LBA 0 is present, merely display it

if ($first_lba == 0)
  {
  print "***** LBA 0: Protective/Legacy MBR *****\n\n";

# Print 31 lines of 16 bytes each, followed by one line of 14 bytes

  for ($i=0; $i<31; $i++)
    {
    $bpos = $i * 16;
    set_output ($bpos,16,@header);
    }

  $bpos += 16;
  set_output ($bpos,14,@header);
  print "\n";

# Process the final two-byte signature (should be 0xAA-55)

  $bpos += 14; $len = 2;
  $temp = &little_endian ($bpos,$len,@header);
  printf ("%03d-%03d  Signature: ", $bpos,$bpos+$len-1);
  list_hex_digits ($bpos,$len,@header);

  if ($temp == 0xaa55)
      { print " [valid]\n\n"; }
    else
      { print " [invalid]\n\n"; }

  $bbase += 512;
  }

# Process LBA 1

$lba = 1;

print "***** LBA 1: Partition Table Header *****\n\n";

# Bytes 0-7 contain the Partition Signature in ASCII

$bpos = 0; $len = 8;
printf ("%03d-%03d  Partition signature: ", $bpos,$bpos+$len-1);
list_hex_digits ($bbase+$bpos, $len, @header);
print " [";
$part_sig = "";
for ($i = $bpos; $i < $bpos+$len; $i++)
  {
  print chr($header[$bbase+$bpos+$i]);
  $part_sig = $part_sig . chr($header[$bbase+$bpos+$i]);
  }

if ($part_sig eq "EFI PART")
    { print "] (Valid signature)\n"; }
  else
    { print "] (Invalid signature)\n"; }

# Bytes 8-11 contain the version number

$bpos = 8; $len = 4;
printf ("%03d-%03d  Version number: ", $bpos,$bpos+$len-1);
list_hex_digits ($bbase+$bpos, $len, @header);
printf (" [v %d.%d]\n", big_endian ($bbase+$bpos,3,@header), big_endian ($bbase+$bpos+3,1,@header));

# Bytes 12-15 contain the header size (should be 92)

$bpos = 12; $len = 4;
printf ("%03d-%03d  Header size: ", $bpos,$bpos+$len-1);
list_hex_digits ($bbase+$bpos, $len, @header);
$temp = little_endian ($bbase+$bpos,$len,@header);
print " [$temp bytes]";
if ($temp != 92)
  { print " (Unexpected value...)"; }
print "\n";

# Bytes 16-19 contain the header CRC32

$bpos = 16; $len = 4;
printf ("%03d-%03d  Header CRC32: ", $bpos,$bpos+$len-1);
list_hex_digits ($bbase+$bpos, $len, @header);
print "\n";

# Skip this next block while working on CRC32 algorithm
goto SKIP_CRC;

$crc_start = (1 - $first_lba) * 512; $crc_len = 16;

for ($i=0; $i<$crc_len; $i++)
  { $temp2 [$i] = $header [$crc_start + $i]; }
$temp = crc32 (@temp2);
printf ("\nCRC = %08X\n", $temp);
SKIP_CRC:

# Bytes 20-23 are reserved and should be 0

$bpos = 20; $len = 4;
printf ("%03d-%03d  (Reserved): ", $bpos,$bpos+$len-1);
list_hex_digits ($bbase+$bpos, $len, @header);
print "\n";

# Bytes 24-31 are the LBA address of the current header (should be 1)

$bpos = 24; $len = 8;
printf ("%03d-%03d  Primary GPT LBA: ", $bpos,$bpos+$len-1);
list_hex_digits ($bbase+$bpos, $len, @header);
$temp = little_endian ($bbase+$bpos,$len,@header);
print " [", commify ($temp), "]\n";

# Bytes 32-39 are the LBA address of the backup header

$bpos = 32; $len = 8;
printf ("%03d-%03d  Backup GPT LBA: ", $bpos,$bpos+$len-1);
list_hex_digits ($bbase+$bpos, $len, @header);
$temp = little_endian ($bbase+$bpos,$len,@header);
print " [", commify ($temp), "]\n";

# Bytes 40-47 are the LBA address of the first useable LBA

$bpos = 40; $len = 8;
printf ("%03d-%03d  First usable LBA: ", $bpos,$bpos+$len-1);
list_hex_digits ($bbase+$bpos, $len, @header);
$temp = little_endian ($bbase+$bpos,$len,@header);
print " [", commify ($temp), "]\n";

# Bytes 48-55 are the LBA address of the backup header

$bpos = 48; $len = 8;
printf ("%03d-%03d  Last useable LBA: ", $bpos,$bpos+$len-1);
list_hex_digits ($bbase+$bpos, $len, @header);
$temp = little_endian ($bbase+$bpos,$len,@header);
print " [", commify ($temp), "]\n";

# Bytes 56-71 are the disk GUID

$bpos = 56; $len = 16;
printf ("%03d-%03d  Disk GUID: ", $bpos,$bpos+$len-1);
list_hex_digits ($bbase+$bpos, $len, @header);
$guid = get_guid ($bbase+$bpos,@header);
print "\n";
print "         GUID: $guid\n";

# Bytes 72-79 are the LBA address of where partition entries start

$bpos = 72; $len = 8;
printf ("%03d-%03d  Partition entries LBA: ", $bpos,$bpos+$len-1);
list_hex_digits ($bbase+$bpos, $len, @header);
$temp = little_endian ($bbase+$bpos,$len,@header);
print " [", commify ($temp), "]\n";

# Bytes 80-83 are the number of partition entries

$bpos = 80; $len = 4;
printf ("%03d-%03d  Number of partition entries: ", $bpos,$bpos+$len-1);
list_hex_digits ($bbase+$bpos, $len, @header);
$num_part_tables = little_endian ($bbase+$bpos,$len,@header);
print " [", commify ($num_part_tables), "]\n";

# Bytes 84-87 are the size of partition entries

$bpos = 84; $len = 4;
printf ("%03d-%03d  Partition entry size: ", $bpos,$bpos+$len-1);
list_hex_digits ($bbase+$bpos, $len, @header);
$part_size = little_endian ($bbase+$bpos,$len,@header);
print " [", commify ($part_size), " bytes]\n";

# Bytes 88-91 contain the partition array CRC32

$bpos = 88; $len = 4;
printf ("%03d-%03d  Partition array CRC32: ", $bpos,$bpos+$len-1);
list_hex_digits ($bbase+$bpos, $len, @header);
print "\n";

# Bytes 92-511 are reserved and must be 0

$bpos = 92; $len = 420;
printf ("%03d-%03d  Reserved section:\n", $bpos,$bpos+$len-1);

# Print 26 lines of 16 bytes each, followed by one line of 4 bytes

for ($i = 92; $i < 512; $i++)
  { $temp2 [$i] = $header [$bbase+$i]; }

for ($i=0; $i<26; $i++)
  {
  $bpos = $i * 16 + 92;
  set_output ($bpos,16,@temp2);
  }

$bpos += 16;
set_output ($bpos,4,@temp2);
print "\n";

# Now process the partition tables in LBA 2-33

# This program assumes the standard partion size of 128 B. If it is
# not 128 that size, skip the processing and end the program.

if ($part_size != 128)
    {
    print "Partition size = $part_size which is non-standard.\n";
    print "Partition table processing skipped...\n";
    goto DONE;
    }

for ($k2 = 1; $k2 <= $num_part_tables; $k2++)
  {

# There are four partition tables per sectors...
# If this is the first table in a given LBA, reset the LBA counter
# In any case, reset counters at the beginning of processing a
#  partition table entry.

  if ($k2%4 == 1)
    {
    $lba++;
    $bbase = $lba * 512;
    }
  $bpos = ($k2-1)%4 * 128;

# If we don't have an entire other entry (i.e., $part_size bytes) left,
# just get out..

  if (($bbase + $bpos + $part_size) > $header_len)
    { goto DONE; }

  if ($k2 == 1)
    { print "***** LBA 2-33: Partition Tables *****\n"; }

  print "\n=== Partition Table #$k2 (LBA $lba, bytes $bpos:", $bpos+$part_size-1, ") ===\n";

# If entire PT entry is zero-filled, don't parse

  $zero = 0;
  for ($i = 0; $i < $part_size; $i++)
    {
    if ($header [$bbase+$bpos+$i] != 0x00)
      { $zero = 1;
        $i = $part_size;
      }
    }

  if ($zero == 0)
    {
    print "Partition table entry is zero-filled\n";
    goto END_OF_PT;
    }

# Bytes 0-15 of the PT entry are the partition type GUID

  $len = 16;
  printf ("%03d-%03d  Partition type GUID: ", $bpos,$bpos+$len-1);
  list_hex_digits ($bbase+$bpos, $len, @header);
  $guid = get_guid ($bbase+$bpos,@header);
  print "\n";
  print "         GUID: $guid\n";
  print "         Type: ";
  if (defined $partition_type{$guid})
      { print "$partition_type{$guid}\n"; }
    else
      { print "(Not listed)\n"; }

# Bytes 16-31 of the PT entry are the unique partition GUID

  $bpos += $len;
  $len = 16;
  printf ("%03d-%03d  Partition GUID: ", $bpos,$bpos+$len-1);
  list_hex_digits ($bbase+$bpos, $len, @header);
  $guid = get_guid ($bbase+$bpos,@header);
  print "\n";
  print "         GUID: $guid\n";

# Bytes 32-39 of the PT entry are the first LBA

  $bpos += $len;
  $len = 8;
  printf ("%03d-%03d  First LBA: ", $bpos,$bpos+$len-1);
  list_hex_digits ($bbase+$bpos, $len, @header);
  $temp = little_endian ($bbase+$bpos,$len,@header);
  print " [", commify ($temp), "]\n";

# Bytes 40-47 of the PT entry are the last LBA

  $bpos += $len;
  $len = 8;
  printf ("%03d-%03d  Last LBA: ", $bpos,$bpos+$len-1);
  list_hex_digits ($bbase+$bpos, $len, @header);
  $temp = little_endian ($bbase+$bpos,$len,@header);
  print " [", commify ($temp), "]\n";

# Bytes 48-55 of the PT entry are the partition attributes

  $bpos += $len;
  $len = 8;
  printf ("%03d-%03d  Partition attributes: ", $bpos,$bpos+$len-1);
  list_hex_digits ($bbase+$bpos, $len, @header);
  print "\n";

# Bytes 56-127 of the PT entry are the partition name (UTF-16LE encoding)

  $bpos += $len;
  $len = 72;
  $epos = $bpos + $len - 1;
  printf ("%03d-%03d  Partition name --\n", $bpos,$epos);

# Print 4 lines of 16 bytes each, followed by one line of 8 bytes

  $pbase = ($k2-1)%4 * $part_size;

  for ($i = $bpos; $i < $epos; $i++)
    { $temp2 [$i] = $header [$bbase + $i]; }

  for ($i=0; $i<4; $i++)
    {
    $bp2 = $pbase + $i * 16 + 56;
    set_output ($bp2,16,@temp2);
    }

  $bp2 += 16;
  set_output ($bp2,8,@temp2);

  print "      Name: ";
  for ($i = 0; $i < $len; $i+=2)
    {
    $temp = $header [$bbase+$bpos+$i] + $header [$bbase+$bpos+$i+1]*256;
    print chr($temp);
    }
  print "\n";

END_OF_PT:
  }

# Finish up!

DONE:

print "\nDone!\n";


# ********************************************************
# *****           SUBROUTINE BLOCKS                  *****
# ********************************************************

# ********************************************************
# big_endian
# ********************************************************

# Given an array name, starting position, and number of digits,
# calculate the decimal value using big endian (MSB first)

sub big_endian
{
my ($start,$n,@array) = @_;
my ($i, $sum);

$sum = $array[$start+$n-1];
for ($i=$n; $i>1; $i--)
  { $sum += $array[$start+$n-$i] * (2**(($i-1)*8)); }

return $sum;
}

# ********************************************************
# commify
# ********************************************************

# Routine from PerlFAQ5 -- add commas to large integers
# http://perldoc.perl.org/perlfaq5.html
  #How-can-I-output-my-numbers-with-commas-added?

sub commify

{
local $_  = shift;
1 while s/^([-+]?\d+)(\d{3})/$1,$2/;
return $_;
}


# ********************************************************************
# CRC32 module by Eli (5/19/2011)
# http://billauer.co.il/blog/2011/05/perl-crc32-crc-xs-module/
#
# See also http://www.greenend.org.uk/rjk/2004/crc.html and
# http://en.wikipedia.org/wiki/CRC32
#
# Comments and modifications by GCK, 6/13/2011. The original routine
# was designed to accept an input string. The problem was that it
# would not handle input that was a set of raw hex bytes. The routine
# was modified to expect an array of hex digits rather than a
# character string. 
# ********************************************************************

sub crc32
{
my (@input, $init_value, $polynomial) = @_;

$init_value = 0 unless (defined $init_value);
$polynomial = 0xedb88320 unless (defined $polynomial);

# @input is an array containing a series of bytes containing hex values

# $init_value is the initial value with which to start the CRC calculation.
# Unless otherwise specified, the CRC calculation starts at 0.

# $polymonial is a hex value that defines the CRC32 polynomial to use. In this
# context, the hex value represents the bits that are on. The default value for
# this routine is the CRC32 used in the IEEE 802.3, Ethernet, and other standards.
# In reversed notation, 0xEDB88320 = 1 0000 0100 1100 0001 0001 1101 1011 0111
# (note the implied leading "1" = x^32 + x^26 + x^23 + x^22 + x^16 + x^12 + x^11
# + x^10 + x^8 + X^7 + X^5 + X^4 + X^2 + X + 1
#
# In normal -- more intutive -- notation, this polynomial would be 0X04C11DB7 

my @lookup_table;
my ($crc, $i, $j, $len, $x);

# Build a lookup table to perform byte-wise calculations

for ($i=0; $i<256; $i++)
  {
  $x = $i;
  for ($j=0; $j<8; $j++)
    {
    if ($x & 1)
        { $x = ($x >> 1) ^ $polynomial; }
      else
        { $x = $x >> 1; }
    }
  push @lookup_table, $x;
  }

# Set the initial value for $crc, then cycle thru the $input array

$crc = $init_value ^ 0xffffffff;

$len = $#input + 1;
for ($i=0; $i<$len; $i++)
  {
  $x = $input [$i];
  $crc = (($crc >> 8) & 0xffffff) ^ $lookup_table[ ($crc ^ $x) & 0xff ];
  }

$crc = $crc ^ 0xffffffff;

return $crc;
}

# ********************************************************
# get_guid
# ********************************************************

# Given an array name and starting position, grab the next 16
# bytes to form the GUID string

sub get_guid
{
my ($start,@array) = @_;
my ($guid_string, $i);

# The GUID format is not straighforward. The GUID is 16 bytes,
# (32 hex digits), generally written in the format:
# xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx

# The first three blocks are byte-swapped, however, so the 16-byte
# sequence 28 73 2A C1 1F F8 D2 11 BA 4B 00 A0 C9 3E C9 3B would
# result in the string C12A7328-F81F-11D2-BA4B-00A0C93EC93B

$guid_string = "";

# Swap bytes 0-3

for ($i = 3; $i >= 0; $i--)
  { $guid_string = $guid_string . sprintf ("%02X", $array[$start+$i]); }
$guid_string = $guid_string . "-";

# Swap bytes 4-5, then 6-7

$guid_string = $guid_string . sprintf ("%02X", $array[$start+5]);
$guid_string = $guid_string . sprintf ("%02X", $array[$start+4]);
$guid_string = $guid_string . "-";
$guid_string = $guid_string . sprintf ("%02X", $array[$start+7]);
$guid_string = $guid_string . sprintf ("%02X", $array[$start+6]);
$guid_string = $guid_string . "-";

# Now grab bytes 8-15 (adding a "-" after 9)

for ($i = 8; $i < 16; $i++)
  { 
  $guid_string = $guid_string . sprintf ("%02X", $array[$start+$i]);
  if ($i == 9)
    { $guid_string = $guid_string . "-"; }
  }

return $guid_string;
}

# ********************************************************
# help_text
# ********************************************************

# Display the help file

sub help_text
{
print<< "EOT";
Program usage: gptparser [-i input_file]
               gptparser {[-h] | [-v] | [-t]}

 where: -i is the input file name
        -h prints this help file
        -v displays the program version number
        -t prints the GPT template file

If no -i switch is provided, the program will ask for the
input file name.

The input file should contain the contents of a GUID Partition.
Table (GPT) header. The file should be a binary file that is
generally 34 sectors (17,408 bytes) in length. The best way to
obtain the GPT is to export the file from your forensics software.

EOT
return;
}


# ********************************************************
# list_hex_digits
# ********************************************************

# Given an array name, starting position, and number of digits, print
# out a string of hex digits in the format 0xAA-BB-CC...

sub list_hex_digits
{
my ($start,$n,@array) = @_;
my $i;

for ($i = $start; $i < $start+$n; $i++)
  {
  if ($i == $start)
      { print "0x"; }
    else
      { print "-"; }
  printf ("%02X", $array[$i]);
  }

return;
}


# ********************************************************
# little_endian
# ********************************************************

# Given an array name, starting position, and number of digits,
# calculate the decimal value using little endian (LSB first)

sub little_endian
{
my ($start,$n,@array) = @_;
my ($i, $sum);

$sum = $array[$start];
for ($i=1; $i<$n; $i++)
  { $sum += $array[$start+$i] * (2**($i*8)); }

return $sum;
}


# ********************************************************
# parsecommandline
# ********************************************************

# Parse command line for file name, if present. Query
# user for any missing information

# Return $state = 1 to indicate that the program should stop
# immediately (switch -h)

sub parsecommandline
{
my $state = 0;

# Parse command line switches ($ARGV array of length $#ARGV)

if ($#ARGV >= 0)

  { 
  for ($i = 0; $i <= $#ARGV; $i++)
    {
    PARMS:
      {
      $ARGV[$i] eq "-i" && do
         {
         $source = $ARGV[$i+1];
         $i++;
         last PARMS;
         };
      $ARGV[$i] eq "-t" && do
         {
         $layout = "layout_GPT.txt";
         open (INFILE, "<", $layout)
          or die "Can't open input file \"$layout\": $!";
         while (not (eof INFILE))
           {
           $line = <INFILE>;
           print $line;
           }
         close INFILE;
         $state = 1;
         return $state;
         };
      $ARGV[$i] eq "-v" && do
         {
         $state = 1;
         return $state;
         };
      $ARGV[$i] eq "-h" && do
         {
         help_text();
         $state = 1;
         return $state;
         };

      do
         {

         print "Invalid parameter \"$ARGV[$i]\".\n\n";
         print "Usage: gptparser [-i input_file]\n";
         print "       gptparser {[-h] | [-v] | [-t]}\n\n";
         $state = 1;
         return $state;
         };
      };
    };
  };

# Prompt for file names, if not supplied

if ($source eq "")

  {
  print "Enter the input file name: ";
  chomp ($source = <STDIN>);
  print "\n";
  };

return $state;
}


# ********************************************************
# set_output
# ********************************************************

# Print output of the data bytes to show both hex and ASCII
# values.

sub set_output
{
my ($bpos,$n,@array) = @_;
my ($j, $k, $linemax);

# Format for a maximum of 16 items per line

$linemax = 16;


printf ("%03d: ", $bpos);
for ($j=0; $j<$n; $j++)
  {
  printf (" %02X", $array[$bpos+$j]);
  }
if ($n < $linemax)
  {
  for ($j=1; $j<=$linemax-$n; $j++)
    { print "   "; };
  }

print "   ";
for ($j=0; $j<$n; $j++)
  {
  $k = $array[$bpos+$j];

#   Print a dot (.) for unprintable characters

  if ($k <= 0x1f || ($k >= 0x7f && $k <= 0x81)
     || ($k >= 0x8d && $k <= 90)
     || $k == 0x9d || $k == 0x9e)
      { print "."; }
    else
      { print chr($k); }
  }
print "\n";

return;
}


